import africastalking from 'africastalking';
import axios from 'axios';
import dotenv from 'dotenv';
dotenv.config();

// ✅ Initialize Africa's Talking SDK
const at = africastalking({
  apiKey: process.env.AFRICASTALKING_API_KEY,
  username: process.env.AFRICASTALKING_USERNAME,
});

const sms = at.SMS;

const AFRICASTALKING_API_KEY = process.env.AFRICASTALKING_API_KEY;
const AFRICASTALKING_USERNAME = process.env.AFRICASTALKING_USERNAME;

/**
 * Send SMS using Africa's Talking
 * @param {string|string[]} to - recipient phone number(s)
 * @param {string} message - message text
 */
export const sendSMS = async (to, message) => {
  try {
    const response = await sms.send({
      to,
      message,
      from: 'Cabral', // must be pre-approved sender ID
    });
    console.log('✅ SMS sent:', response);
    return response;
  } catch (err) {
    console.error('❌ Failed to send SMS:', err.message);
    throw err;
  }
};

/**
 * Send voice call using Africa's Talking Voice API
 * @param {string|string[]} to - single phone number or array of numbers
 * @param {string} from - approved number or sender ID
 * @param {string} audioUrl - public URL of MP3/voice file
 */
export const sendVoiceMail = async (to, from, audioUrl) => {
  const url = 'https://voice.africastalking.com/call';

  if (!Array.isArray(to)) {
    to = [to];
  }

  const payload = {
    username: AFRICASTALKING_USERNAME,
    from,
    to,
    callActions: [
      {
        actionType: 'Play',
        url: audioUrl,
      },
    ],
  };

  console.log('📤 Sending voice call with payload:', payload);

  try {
    const response = await axios.post(url, payload, {
      headers: {
        Accept: 'application/json',
        apiKey: AFRICASTALKING_API_KEY,
        'Content-Type': 'application/json',
      },
    });

    console.log('✅ Voice call initiated:', response.data);
    return response.data;
  } catch (error) {
    console.error('❌ Failed to send voice call:', error)
    return {
      status: 'error',
      message: error.message,
      details: error.response?.data || null,
    };
  }
};

/**
 * Check voice queue status for phone numbers
 * @param {string} phoneNumbers - comma-separated string of numbers
 */
export const getQueueStatus = async (phoneNumbers) => {
  const url = 'https://voice.africastalking.com/queueStatus';

  const payload = {
    username: AFRICASTALKING_USERNAME,
    phoneNumbers,
  };

  console.log('📥 Checking queue status for:', phoneNumbers);

  try {
    const response = await axios.post(url, payload, {
      headers: {
        Accept: 'application/json',
        apiKey: AFRICASTALKING_API_KEY,
      },
    });

    console.log('✅ Queue status response:', response.data);
    return response.data;
  } catch (error) {
    console.error('❌ Failed to fetch queue status:', error.message);
    return {
      status: 'error',
      message: error.message,
      details: error.response?.data || null,
    };
  }
};
