import multer from 'multer';
import path from 'path';
import fs from 'fs';

// Define the uploads directory
const UPLOAD_DIR = 'uploads/';

// Ensure uploads folder exists
if (!fs.existsSync(UPLOAD_DIR)) {
  fs.mkdirSync(UPLOAD_DIR, { recursive: true });
}

// ✅ 1. Memory Storage (for CSV/XLSX/JSON file import, kept in memory)
export const upload = multer({
  storage: multer.memoryStorage(),
  limits: {
    fileSize: 10 * 1024 * 1024, // 10 MB limit
  },
});

// ✅ 2. Disk Storage (used for uploading actual audio files)
const diskStorage = multer.diskStorage({
  destination: (req, file, cb) => {
    cb(null, UPLOAD_DIR);
  },
  filename: (req, file, cb) => {
    const ext = path.extname(file.originalname);
    const baseName = path.basename(file.originalname, ext);
    const timestamp = Date.now();
    const sanitized = baseName.replace(/\s+/g, '-').toLowerCase(); // optional: clean filename
    cb(null, `${sanitized}-${timestamp}${ext}`);
  },
});

// ✅ 3. Audio File Filter (only allow audio MIME types)
const audioFilter = (req, file, cb) => {
  if (file.mimetype.startsWith('audio/')) {
    cb(null, true);
  } else {
    cb(new Error('Only audio files are allowed (e.g., mp3, wav)'), false);
  }
};

// ✅ 4. Export disk-based audio uploader with filter
export const uploadAudio = multer({
  storage: diskStorage,
  fileFilter: audioFilter,
  limits: {
    fileSize: 20 * 1024 * 1024, // 20 MB max for audio
  },
});
