import Delegate from "../models/Delegate.js";
import Leader from "../models/Leader.js";
import Permission from "../models/Permission.js";
import Role from "../models/Role.js";
import Setting from "../models/Setting.js";
import User from "../models/User.js";
import backup from "mongodb-backup";
import path from "path";
import fs from "fs";
import archiver from "archiver";
import { exec } from "child_process";
import { Parser } from "json2csv";

export const getSystemSettings = async (req, res) => {
    try {
        let settings = await Setting.findOne({ type: "system" });
        if (!settings) settings = { system: {} };
        res.json(settings.system);
    } catch (error) {
        console.error("Error fetching system settings:", error);
        res.status(500).json({ error: "Failed to fetch system settings" });
    }
};

export const saveSystemSettings = async (req, res) => {
    try {
        const data = req.body;
        let settings = await Setting.findOneAndUpdate(
            { type: "system" },
            { system: data },
            { new: true, upsert: true }
        );
        res.json(settings.system);
    } catch (error) {
        console.error("Error saving system settings:", error);
        res.status(500).json({ error: "Failed to save system settings" });
    }
};

export const exportAllData = async (req, res) => {
    try {
        const collections = {
            delegates: await Delegate.find().lean(),
            leaders: await Leader.find().lean(),
            permissions: await Permission.find().lean(),
            roles: await Role.find().lean(),
            settings: await Setting.find().lean(),
            users: await User.find().lean(),
        };

        res.setHeader("Content-Type", "application/zip");
        res.setHeader(
            "Content-Disposition",
            "attachment; filename=all-data.zip"
        );

        const archive = archiver("zip", { zlib: { level: 9 } });

        archive.pipe(res);

        const json2csv = (data) => {
            const parser = new Parser();
            return parser.parse(data);
        };

        for (const [name, data] of Object.entries(collections)) {
            const csv = json2csv(data);
            console.log(csv);

            archive.append(csv, { name: `${name}.csv` });
        }

        await archive.finalize();
    } catch (error) {
        console.error("Export error:", error);
        res.status(500).json({ error: "Failed to export data" });
    }
};

export const backupDatabase = async (req, res) => {
    try {
        const backupDir = path.join(process.cwd(), "backups");
        if (!fs.existsSync(backupDir)) fs.mkdirSync(backupDir);

        const backupPath = path.join(backupDir, `db-backup-${Date.now()}.gz`);

        backup({
            uri: process.env.MONGO_BACKUP_URI,
            root: backupPath,
            gzip: true,
        });

        console.log(backupPath);

        res.json({
            message: "Database backup completed",
            url: `/backups/${path.basename(backupPath)}`,
        });
    } catch (error) {
        console.error("Backup error:", error);
        res.status(500).json({ error: "Failed to backup database" });
    }
};
