import Payment from "../models/Payment.js";
import Leader from "../models/Leader.js";
import { makePayout } from "../utils/flutterwave.js";
import { v4 as uuidv4 } from "uuid";



export const initiatePayoutToSingleLeader = async (req, res) => {
  try {
    const { amount, paymentMethod, description, recipients, currency = "UGX" } = req.body;

    if (!recipients || !Array.isArray(recipients) || recipients.length === 0) {
      return res.status(400).json({ error: "No recipient data provided." });
    }

    const { leaderId, phone } = recipients[0];

    if (!leaderId || !amount) {
      return res.status(400).json({ error: "Leader ID and amount are required." });
    }

    const leader = await Leader.findById(leaderId);
    if (!leader) {
      return res.status(404).json({ error: "Leader not found." });
    }

    const reference = `single_ref_${uuidv4().slice(0, 8)}_${leader._id}`;
    let payoutStatus = "failed";
    let transactionId = null;

    try {
      const payoutRes = await makePayout({
        account_bank: leader.bankCode || "MPS",
        account_number: leader.accountNumber || phone || leader.phone,
        beneficiary_name: `${leader.firstName} ${leader.secondName}`,
        amount,
        currency,
        narration: description || `Payout to ${leader.firstName} ${leader.secondName}`,
        reference,
      });

      payoutStatus = payoutRes.success ? "sent" : "failed";
      transactionId = payoutRes?.data?.id || null;
    } catch (payoutError) {
      console.error(`Payout failed for leader ${leader._id}:`, payoutError.message);
    }

    const paymentDoc = await Payment.create({
      description: description || `Payout to ${leader.firstName} ${leader.secondName}`,
      amount,
      method: paymentMethod || "flutterwave-payout",
      currency,
      category: leader.category,
      specialInterestGroup: leader.specialInterestGroup || null,
      leadershipLevel: leader.leadershipLevel || null,
      createdBy: req.user._id,
      status: payoutStatus,
      recipients: [
        {
          leader: leader._id,
          amount,
          reference,
          status: payoutStatus,
          transactionId,
          processedAt: payoutStatus === "sent" ? new Date() : null,
        },
      ],
    });

    res.status(200).json({
      message: "Single payout completed",
      status: payoutStatus,
      payment: paymentDoc,
    });
  } catch (err) {
    console.error("Single payout error:", err);
    res.status(500).json({ error: "Single payout failed" });
  }
};

/**
 * Initiate Payouts via Flutterwave
 */
export const initiatePayoutToLeaders = async (req, res) => {
  const {
    amount,
    category,
    leadershipLevel,
    specialInterestGroup,
    description,
    currency = "UGX",
  } = req.body;

  console.log("Initiating payout with data:", {
    amount,
    category,
    leadershipLevel,
    specialInterestGroup,
    description,
    currency,
  });

  try {
    const query = { category };
    if (category === "special-interest") {
      query.specialInterestGroup = specialInterestGroup;
    }
    if (leadershipLevel) {
      query.leadershipLevel = leadershipLevel;
    }

    const leaders = await Leader.find(query);
    if (leaders.length === 0) {
      return res.status(400).json({ error: "No eligible leaders found" });
    }

    const paymentDoc = new Payment({
      description,
      amount: amount * leaders.length,
      method: "flutterwave-payout",
      currency,
      category,
      specialInterestGroup: specialInterestGroup || null,
      leadershipLevel: leadershipLevel || null,
      createdBy: req.user._id,
      status: "pending",
      recipients: [],
    });

    for (const leader of leaders) {
      const reference = `ref_${uuidv4().slice(0, 8)}_${leader._id}`;
      try {
        const payoutRes = await makePayout({
          account_bank: leader.bankCode || "MPS",
          account_number: leader.accountNumber || leader.phone,
          beneficiary_name: `${leader.firstName} ${leader.secondName}`,
          amount,
          currency,
          narration: `Payout to ${leader.firstName} ${leader.secondName}`,
          reference,
        });

        paymentDoc.recipients.push({
          leader: leader._id,
          amount,
          reference,
          status: payoutRes.success ? "sent" : "failed",
          transactionId: payoutRes?.data?.id || null,
          processedAt: payoutRes.success ? new Date() : null,
        });
      } catch (payoutError) {
        console.error(`Payout failed for ${leader._id}:`, payoutError.message);
        paymentDoc.recipients.push({
          leader: leader._id,
          amount,
          reference,
          status: "failed",
          transactionId: null,
        });
      }
    }

    const allSuccess = paymentDoc.recipients.every(r => r.status === "sent");
    const anySuccess = paymentDoc.recipients.some(r => r.status === "sent");

    paymentDoc.status = allSuccess
      ? "sent"
      : anySuccess
      ? "partially-sent"
      : "failed";

    await paymentDoc.save();

    res.status(200).json({
      message: "Payout process complete",
      status: paymentDoc.status,
      recipients: paymentDoc.recipients,
    });
  } catch (err) {
    console.error("Payout error:", err);
    res.status(500).json({ error: "Payout setup failed" });
  }
};

/**
 * Handle Flutterwave payout callback
 */
export const handlePayoutCallback = async (req, res) => {
  try {
    const callbackData = req.body;
    console.log("Payout callback received:", JSON.stringify(callbackData, null, 2));

    const {
      status,             // 'SUCCESSFUL', 'FAILED', 'PENDING'
      id: transactionId,
      reference,
    } = callbackData?.data || {};

    if (!reference || !transactionId || !status) {
      return res.status(400).json({ error: "Invalid callback payload" });
    }

    const payment = await Payment.findOne({ "recipients.reference": reference });
    if (!payment) {
      return res.status(404).json({ error: "Payment with this reference not found" });
    }

    const updatedRecipients = payment.recipients.map((recipient) => {
      if (recipient.reference === reference) {
        return {
          ...recipient.toObject(),
          status: status.toLowerCase(),
          transactionId,
          processedAt: new Date(),
        };
      }
      return recipient;
    });

    const allSent = updatedRecipients.every(r => r.status === "successful");
    const anySent = updatedRecipients.some(r => r.status === "successful");

    payment.recipients = updatedRecipients;
    payment.status = allSent
      ? "sent"
      : anySent
      ? "partially-sent"
      : "failed";

    await payment.save();

    return res.status(200).json({ message: "Payout updated", status: payment.status });
  } catch (err) {
    console.error("Error handling payout callback:", err.message);
    return res.status(500).json({ error: "Callback processing failed" });
  }
};

/**
 * Get all payments
 */
export const getPayments = async (req, res) => {
  try {
    const { status, createdBy, limit = 100, page = 1 } = req.query;

    const query = {};
    if (status) query.status = status;
    if (createdBy) query.createdBy = createdBy;

    const payments = await Payment.find(query)
      .sort({ createdAt: -1 })
      .skip((page - 1) * limit)
      .limit(parseInt(limit))
      .populate("recipients.leader", "firstName secondName phone gender position");

    const total = await Payment.countDocuments(query);

    res.json({
      data: payments,
      meta: {
        total,
        page: parseInt(page),
        limit: parseInt(limit),
        pages: Math.ceil(total / limit),
      },
    });
  } catch (err) {
    console.error("Get payments error:", err);
    res.status(500).json({ error: "Failed to fetch payments" });
  }
};

/**
 * Delete all payments
 */
export const deleteAllPayments = async (req, res) => {
  try {
    const result = await Payment.deleteMany({});
    res.json({
      message: "All payments deleted successfully",
      deletedCount: result.deletedCount,
    });
  } catch (err) {
    console.error("Delete all payments error:", err);
    res.status(500).json({ error: "Failed to delete payments" });
  }
};


export const getPaymentSummary = async (req, res) => {
  try {
    const payments = await Payment.find();

    let totalAmount = 0;
    let recipientsPaid = 0;
    let pendingPayments = 0;
    let failedPayments = 0;

    payments.forEach((payment) => {
      payment.recipients.forEach((recipient) => {
        totalAmount += recipient.amount;

        if (recipient.status === "success" || recipient.status === "sent") {
          recipientsPaid += 1;
        } else if (recipient.status === "pending") {
          pendingPayments += 1;
        } else if (recipient.status === "failed") {
          failedPayments += 1;
        }
      });
    });

    return res.json({
      success: true,
      data: {
        totalDistributedUGX: totalAmount,
        recipientsPaid,
        pendingPayments,
        failedPayments,
      },
    });
  } catch (error) {
    console.error("Payment summary error:", error);
    return res.status(500).json({
      success: false,
      message: "Failed to fetch payment summary",
    });
  }
};


export const getRecipientsCount = async (req, res) => {
  try {
    const { category, leadershipLevel, specialInterestGroup } = req.query;

    if (!category) {
      return res.status(400).json({ error: "Category is required" });
    }

    const filter = {};

    // Filter by category
    if (category === "special-interest") {
      filter.category = "special-interest";
      if (specialInterestGroup) {
        filter.specialInterestGroup = specialInterestGroup;
      }
    } else {
      filter.category = "mainstream";
      if (leadershipLevel && leadershipLevel !== "all") {
        filter.leadershipLevel = new RegExp(`^${leadershipLevel}$`, "i");
      }
    }

    const count = await Leader.countDocuments(filter);

    return res.json({ count });
  } catch (err) {
    console.error("Error in getRecipientsCount:", err);
    return res.status(500).json({ error: "Server error" });
  }
};