import Delegate from '../models/Delegate.js';
import fs from "fs";
import xlsx from "xlsx";
// Create/Register
export const registerDelegate = async (req, res) => {
  try {
    const {
      firstName,
      secondName,
      nin,
      phone,
      gender,
      district,
      subCounty,
      committee,
      position,
    } = req.body;

    const created_by = req.user?.id || req.body.created_by;

    if (
      !firstName || !secondName ||
      !district || !committee || !position
    ) {
      return res.status(400).json({ error: 'All required fields must be filled, including created_by.' });
    }

    const exists = await Delegate.findOne({ nin });
    if (exists) {
      return res.status(409).json({ error: 'A delegate with this NIN already exists.' });
    }

    const delegate = new Delegate({
      firstName,
      secondName,
      nin,
      phone,
      gender,
      district,
      subCounty,
      committee,
      position,
      created_by,
    });

    await delegate.save();
    return res.status(201).json({ message: 'Delegate registered successfully', data: delegate });
  } catch (err) {
    console.error(err);
    res.status(500).json({ error: 'Server error' });
  }
};

// Update
export const updateDelegate = async (req, res) => {
  try {
    const { id } = req.params;
    const updated = await Delegate.findByIdAndUpdate(id, req.body, { new: true });
    if (!updated) {
      return res.status(404).json({ error: 'Delegate not found' });
    }
    res.status(200).json({ message: 'Delegate updated', data: updated });
  } catch (err) {
    res.status(500).json({ error: 'Update failed' });
  }
};

// Get All
export const getAllDelegates = async (req, res) => {
  try {
    const delegates = await Delegate.find().sort({ createdAt: -1 });
    res.status(200).json({ data: delegates });
  } catch (err) {
    res.status(500).json({ error: 'Could not fetch delegates' });
  }
};

// Get by ID
export const getDelegateById = async (req, res) => {
  try {
    const { id } = req.params;
    const delegate = await Delegate.findById(id);
    if (!delegate) {
      return res.status(404).json({ error: 'Delegate not found' });
    }
    res.status(200).json({ data: delegate });
  } catch (err) {
    res.status(500).json({ error: 'Could not retrieve delegate' });
  }
};

// Delete one
export const deleteDelegate = async (req, res) => {
  try {
    const { id } = req.params;
    const deleted = await Delegate.findByIdAndDelete(id);
    if (!deleted) {
      return res.status(404).json({ error: 'Delegate not found' });
    }
    res.status(200).json({ message: 'Delegate deleted' });
  } catch (err) {
    res.status(500).json({ error: 'Delete failed' });
  }
};

// Delete all
export const deleteAllDelegates = async (req, res) => {
  try {
    await Delegate.deleteMany({});
    res.status(200).json({ message: 'All delegates deleted' });
  } catch (err) {
    res.status(500).json({ error: 'Bulk delete failed' });
  }
};


export const updateDelegateStatus = async (req, res) => {
  try {
    const { id } = req.params;
    const { status } = req.body;

    if (!["pending", "verified", "rejected"].includes(status)) {
      return res.status(400).json({ error: "Invalid status value." });
    }

    const updated = await Delegate.findByIdAndUpdate(id, { status }, { new: true });
    if (!updated) return res.status(404).json({ error: "Delegate not found" });

    res.status(200).json({ message: "Status updated", data: updated });
  } catch (err) {
    res.status(500).json({ error: "Failed to update status" });
  }
};


export const getDelegateStats = async (req, res) => {
  try {
    const [total, verified, pending, rejected] = await Promise.all([
      Delegate.countDocuments(),
      Delegate.countDocuments({ status: "verified" }),
      Delegate.countDocuments({ status: "pending" }),
      Delegate.countDocuments({ status: "rejected" }),
    ]);

    res.status(200).json({
      total,
      verified,
      pending,
      rejected,
    });
  } catch (err) {
    res.status(500).json({ error: "Failed to get delegate stats" });
  }
};


export const searchDelegateByUniqueId = async (req, res) => {
  const uniqueId = req.body.uniqueId;

 if (!uniqueId || typeof uniqueId !== "string") {
    return res.status(400).json({ message: "Invalid unique ID." });
  }
  console.log("Searching for delegate with unique ID:", uniqueId);

  try {
    const delegate = await Delegate.findOne({ uniqueCode: uniqueId });

    if (!delegate) {
      return res.status(404).json({ message: "Delegate not found." });
    }

    return res.json(delegate);
  } catch (error) {
    console.error("Search error:", error);
    return res.status(500).json({ message: "Server error." });
  }
};


export const verifyDelegateById = async (req, res) => {
  const { id } = req.params;
  const { action } = req.body;

  if (!["verify", "reject"].includes(action)) {
    return res.status(400).json({ message: "Invalid action." });
  }

  try {
    const delegate = await Delegate.findById(id);

    if (!delegate) {
      return res.status(404).json({ message: "Delegate not found." });
    }

    delegate.status = action === "verify" ? "verified" : "rejected";
    delegate.verifiedAt = new Date();
    await delegate.save();

    return res.json({ message: `Delegate ${action}ed successfully.` });
  } catch (error) {
    console.error("Verification error:", error);
    return res.status(500).json({ message: "Server error." });
  }
};

export const bulkUploadDelegates = async (req, res) => {
  try {
    if (!req.file) {
      return res.status(400).json({ message: "No file uploaded" });
    }

    const workbook = xlsx.readFile(req.file.path);
    const sheet = workbook.Sheets[workbook.SheetNames[0]];
    const rows = xlsx.utils.sheet_to_json(sheet);

    let total = rows.length;
    let successful = 0;
    let failed = 0;
    let errors = [];

    for (let [i, row] of rows.entries()) {
      try {
        const requiredFields = ["firstName", "secondName", "phone", "district", "committee", "position"];
        const missingFields = requiredFields.filter(f => !row[f]);
        if (missingFields.length) {
          throw new Error(`Row ${i + 2}: Missing fields: ${missingFields.join(", ")}`);
        }

        await Delegate.create({
          ...row,
          created_by: req.user?._id || "000000000000000000000000", // adjust as needed
        });
        successful++;
      } catch (err) {
        failed++;
        errors.push(err.message);
      }
    }

    fs.unlinkSync(req.file.path); // remove temp file

    res.json({ total, successful, failed, errors });
  } catch (err) {
    res.status(500).json({ message: "Bulk upload failed", error: err.message });
  }
};