import { sendSMS,sendVoiceMail } from "../utils/smsService.js";
import Leader from "../models/Leader.js";
import Communication from "../models/Communication.js";

export const sendCommunication = async (req, res) => {
  try {
    let {
      category,
      specialInterestGroup,
      district,
      message,
    } = req.body;

    let audioUrl = null;
    const type = message ? 'sms' : 'voice';

    // Handle voice file (already saved by multer)
    if (type === 'voice' && req.file) {
      audioUrl = `/uploads/${req.file.filename}`;
      console.log('📤 Audio file saved at:', audioUrl);
    }

    // Filter leaders
    const filter = {};
    if (district && district !== 'all') {
      filter.district = new RegExp(`^${district}$`, 'i');
    }

    if (category === 'special-interest') {
      filter.category = 'special-interest';
      if (specialInterestGroup) {
        filter.specialInterestGroup = specialInterestGroup;
      }
    } else {
      filter.category = 'mainstream';
    }

    const leaders = await Leader.find(filter);
    const phones = leaders.map((l) => l.phone);
    const recipientIds = leaders.map((l) => l._id);

    if (!phones.length) {
      return res.status(400).json({ error: 'No recipients found for this selection.' });
    }

    let status = 'pending';
    let result = null;

    if (type === 'sms') {
      if (!message) {
        return res.status(400).json({ error: 'Message content is required for SMS' });
      }
      result = await sendSMS(phones, message);
      status = result.SMSMessageData?.Recipients?.length > 0 ? 'delivered' : 'failed';
    } else if (type === 'voice') {
      if (!audioUrl) {
        return res.status(400).json({ error: 'Audio file is required for voice messages' });
      }
     result = await sendVoiceMail(phones, '+256323200862', `${process.env.BASE_URL}${audioUrl}`);

      status = result?.entries?.length > 0 ? 'queued' : 'failed';
    }

    const saved = await Communication.create({
      messageType: type,
      category,
      specialInterestGroup: category === 'special-interest' ? specialInterestGroup : undefined,
      district,
      message: type === 'sms' ? message : undefined,
      audioUrl: type === 'voice' ? audioUrl : undefined,
      sentCount: phones.length,
      recipientIds,
      status,
      created_by: req.user._id,
    });

    return res.status(200).json({
      message: type === 'sms' ? 'SMS sent' : 'Voice message queued',
      result,
      saved,
    });
  } catch (err) {
    console.error('❌ sendCommunication error:', err);
    res.status(500).json({ error: 'Server error', detail: err.message });
  }
};

// 📥 Get Communications
export const getCommunications = async (req, res) => {
  try {
    const { district, type, from, to } = req.query;
    const filter = {};

    if (district && district !== 'all') {
      filter.district = new RegExp(`^${district}$`, 'i');
    }
    if (type) filter.messageType = type;
    if (from || to) {
      filter.createdAt = {};
      if (from) filter.createdAt.$gte = new Date(from);
      if (to) filter.createdAt.$lte = new Date(to);
    }

    const communications = await Communication.find(filter)
      .sort({ createdAt: -1 })
      .limit(100)
      .populate('created_by', 'firstName lastName')
      .populate('recipientIds', 'firstName lastName phone district');

    res.json({ data: communications });
  } catch (err) {
    console.error("❌ getCommunications error:", err);
    res.status(500).json({ error: 'Failed to fetch communications' });
  }
};
