import Role from '../models/Role.js';
import Permission from '../models/Permission.js';
// Create a new role
export const createRole = async (req, res) => {
  const { name, description, permissions } = req.body;

  try {
    // Check if role name already exists
    const existing = await Role.findOne({ name });
    if (existing) {
      return res.status(400).json({ error: 'Role name already exists' });
    }

    // Create new role
    const role = new Role({
      name,
      description,
      permissions: permissions || []
    });

    await role.save();
    const populated = await role.populate('permissions');

    res.status(201).json({
      message: 'Role created successfully',
      role: populated
    });
  } catch (err) {
    console.error('Create role error:', err);
    res.status(500).json({ error: 'Failed to create role' });
  }
};

export const updateRole = async (req, res) => {
  try {
    const { name, description, permissions } = req.body;
    const role = await Role.findById(req.params.id);

    if (!role) {
      return res.status(404).json({ success: false, message: 'Role not found' });
    }

    role.name = name || role.name;
    role.description = description || role.description;

    if (permissions) {
      const resolvedPermissions = await Permission.find({ _id: { $in: permissions } });
      role.permissions = resolvedPermissions.map(p => p._id);
    }

    await role.save();
    const populated = await role.populate('permissions');

    res.status(200).json({ success: true, message: 'Role updated', data: populated });
  } catch (error) {
    console.error('Error updating role:', error);
    res.status(500).json({ success: false, message: 'Server error' });
  }
};


// Get all roles with their permissions
export const getRoles = async (req, res) => {
  try {
    const roles = await Role.find().populate('permissions');
    res.json(roles);
  } catch (err) {
    console.error('Get roles error:', err);
    res.status(500).json({ error: 'Failed to fetch roles' });
  }
};

// (Optional) Get a single role by ID
export const getRoleById = async (req, res) => {
  try {
    const role = await Role.findById(req.params.id).populate('permissions');
    if (!role) return res.status(404).json({ error: 'Role not found' });
    res.json(role);
  } catch (err) {
    console.error('Get role by ID error:', err);
    res.status(500).json({ error: 'Failed to fetch role' });
  }
};

// (Optional) Delete a role
export const deleteRole = async (req, res) => {
  try {
    const result = await Role.findByIdAndDelete(req.params.id);
    if (!result) return res.status(404).json({ error: 'Role not found' });
    res.json({ message: 'Role deleted' });
  } catch (err) {
    console.error('Delete role error:', err);
    res.status(500).json({ error: 'Failed to delete role' });
  }
};
