import Leader from "../models/Leader.js";
import csvParser from "csv-parser";
import { Readable } from "stream";
import XLSX from "xlsx";
// Create
export const createLeader = async (req, res) => {
    try {
        const leader = new Leader({
            ...req.body,
            created_by: req.user._id,
        });
        await leader.save();
        res.status(201).json({ message: "Leader created", data: leader });
    } catch (err) {
        console.error("Create leader error:", err);
        res.status(400).json({ error: err.message });
    }
};

export const getLeaders = async (req, res) => {
    console.log("Get leaders request:", req.query); // DEBUG
    
  try {
    const { category = "mainstream", specialInterestGroup } = req.query;

    const filter = { category };

    if (category == "special-interest" && specialInterestGroup) {
      filter.specialInterestGroup = specialInterestGroup;
    }

    console.log("Leader filter query:", filter); // DEBUG

    const leaders = await Leader.find(filter).sort({ createdAt: -1 });
    res.json(leaders);
  } catch (err) {
    console.error("Get leaders error:", err);
    res.status(500).json({ error: "Server error" });
  }
};


// Get by ID
export const getLeaderById = async (req, res) => {
    try {
        const leader = await Leader.findById(req.params.id);
        if (!leader) return res.status(404).json({ error: "Leader not found" });
        res.json(leader);
    } catch (err) {
        res.status(500).json({ error: "Server error" });
    }
};
export const getLeaderByNin = async (req, res) => {
  try {
    const { nin } = req.params;
    if (!nin) {
      return res.status(400).json({ error: "NIN is required" });
    }

    const leader = await Leader.findOne({ uniqueCode: nin.trim() });
    if (!leader) {
      return res.status(404).json({ error: "Leader not found" });
    }

    res.json(leader);
  } catch (err) {
    console.error("Error fetching leader by NIN:", err);
    res.status(500).json({ error: "Server error" });
  }
};

// Update
export const updateLeader = async (req, res) => {
    try {
        const leader = await Leader.findByIdAndUpdate(req.params.id, req.body, {
            new: true,
        });
        if (!leader) return res.status(404).json({ error: "Leader not found" });
        res.json({ message: "Leader updated", data: leader });
    } catch (err) {
        res.status(400).json({ error: err.message });
    }
};

// Delete
export const deleteLeader = async (req, res) => {
    try {
        const leader = await Leader.findByIdAndDelete(req.params.id);
        if (!leader) return res.status(404).json({ error: "Leader not found" });
        res.json({ message: "Leader deleted" });
    } catch (err) {
        res.status(500).json({ error: "Server error" });
    }
};
export const importLeaders = async (req, res) => {
  try {
    if (!req.file)
      return res.status(400).json({ error: "No file uploaded" });

    const createdBy = req.user._id;
    const leaders = [];
    const skippedRows = [];

    const isExcel = req.file.originalname.endsWith(".xlsx");
    let rows = [];

    if (isExcel) {
      const workbook = XLSX.read(req.file.buffer, { type: "buffer" });
      const sheet = workbook.Sheets[workbook.SheetNames[0]];
      rows = XLSX.utils.sheet_to_json(sheet);
    } else {
      rows = await new Promise((resolve, reject) => {
        const results = [];
        Readable.from(req.file.buffer)
          .pipe(csvParser())
          .on("data", (row) => {
            const values = Object.values(row).map((v) => v?.trim());
            const isEmptyRow = values.every((v) => !v);
            if (!isEmptyRow) results.push(row);
          })
          .on("end", () => resolve(results))
          .on("error", reject);
      });
    }

    for (const row of rows) {
      const {
        firstName,
        secondName,
        gender,
        nin,
        phone,
        leadershipLevel,
        district,
        subCounty,
        parish,
        village,
        position,
        category,
        specialInterestGroup,
      } = row;

      if (
        !firstName ||
        !secondName ||
        !phone ||
        !leadershipLevel ||
        !district
      ) {
        skippedRows.push({ reason: "Missing required fields", row });
        continue;
      }

      try {
        const leaderData = {
          firstName: firstName?.trim(),
          secondName: secondName?.trim(),
          gender: gender?.trim(),
          nin: nin?.trim(),
          phone:normalizePhone(phone),
          leadershipLevel: leadershipLevel?.trim(),
          district: district?.trim(),
          subCounty: subCounty?.trim() || "",
          parish: parish?.trim() || "",
          village: village?.trim() || "",
          position: position?.trim() || "",
          category: category?.trim() || "mainstream",
          created_by: createdBy,
        };

        // Only set specialInterestGroup if the category is "special-interest"
        if (leaderData.category === "special-interest" && specialInterestGroup) {
          leaderData.specialInterestGroup = specialInterestGroup?.trim();
        }

        const leader = new Leader(leaderData);
        await leader.save();
        leaders.push(leader);
      } catch (err) {
        skippedRows.push({ reason: err.message, row });
      }
    }

    res.status(201).json({
      message: "Leader import completed",
      created: leaders.length,
      skipped: skippedRows.length,
      skippedRows,
    });
  } catch (err) {
    console.error("Import error:", err);
    res.status(500).json({ error: "Failed to import leaders" });
  }
};
function normalizePhone(phone) {
  // Convert to string and remove all non-digit characters
  phone = String(phone).replace(/\D/g, '');

  // If starts with 0 and is 10 digits like 0700353769 → remove 0 and add +256
  if (phone.length === 10 && phone.startsWith('0')) {
    return '+256' + phone.slice(1);
  }

  // If starts with 7 and is 9 digits like 700353769 → add +256
  if (phone.length === 9 && phone.startsWith('7')) {
    return '+256' + phone;
  }

  // If already in international format (starts with 256), add +
  if (phone.length === 12 && phone.startsWith('256')) {
    return '+' + phone;
  }

  // Return original if it doesn't match any pattern
  return phone;
}

export const getLeaderStats = async (req, res) => {
  try {
    const pipeline = [
      {
        $match: {
          category: { $regex: /^mainstream$/i },
          leadershipLevel: { $exists: true, $ne: null },
        },
      },
      {
        $group: {
          _id: "$leadershipLevel",
          count: { $sum: 1 },
        },
      },
      {
        $project: {
          leadershipLevel: "$_id",
          count: 1,
          _id: 0,
        },
      },
      {
        $sort: {
          leadershipLevel: 1,
        },
      },
    ];

    const stats = await Leader.aggregate(pipeline);

    // Ensure all levels are represented even if missing
    const levels = ["CEC", "DEC", "SCEC", "PEC", "VEC"];
    const formattedStats = levels.map((level) => {
      const match = stats.find((s) => s.leadershipLevel === level);
      return {
        leadershipLevel: level,
        count: match ? match.count : 0,
      };
    });

    res.status(200).json({ data: formattedStats });
  } catch (err) {
    console.error("❌ Aggregation failed:", err);
    res.status(500).json({
      error: "Server error",
      message: err.message,
    });
  }
};


export const getSpecialInterestLeaderCounts = async (req, res) => {
  try {
    const groups = [
      "youth",
      "elderly",
      "women",
      "veterans",
      "disabled",
      "workers",
      "entrepreneurs",
    ];

  
    // Convert aggregation result into an object with defaults
   const counts = {};
for (const group of groups) {
  try {
    const count = await Leader.countDocuments({
      category: "special-interest",
      specialInterestGroup: group,
    });
    counts[group] = count;
  } catch (err) {
    console.error(`Error counting ${group}:`, err);
    counts[group] = 0;
  }
}
    res.status(200).json({ success: true, data: counts });
  } catch (error) {
    console.error("Error fetching leader counts:", error);
    res.status(500).json({
      success: false,
      error: error.message || "Failed to fetch leader counts",
    });
  }
};