import User from '../models/User.js';
import Role from '../models/Role.js';
import bcrypt from 'bcrypt';

import mongoose from 'mongoose';
export const createUser = async (req, res) => {
  const { name, email, password, role, isActive } = req.body;

  try {
    const existing = await User.findOne({ email });
    if (existing) return res.status(400).json({ error: 'Email already in use' });

    const hashedPassword = password ? await bcrypt.hash(password, 10) :  await bcrypt.hash("123123123", 10);

    const user = new User({
      name,
      email,
      password: hashedPassword,
      role: role,
      isActive
    });

    await user.save();

    // 🔥 Populate role before sending back
    await user.populate('role');

    res.status(201).json({ message: 'User created', user });
  } catch (err) {
    res.status(500).json({ error: err.message });
  }
};


// Get all users
export const getUsers = async (req, res) => {
  try {
    const users = await User.find({ isAdmin: false }).populate('role');
    res.json(users);
  } catch (error) {
    console.error('Error fetching users:', error);
    res.status(500).json({ error: 'Internal server error' });
  }
};

// Update user
export const updateUser = async (req, res) => {
  const { id } = req.params;
  const { name, email, role, isActive } = req.body;

  try {
    if (!mongoose.Types.ObjectId.isValid(role)) {
      return res.status(400).json({ error: 'Malformed role ID' });
    }

    const roledata = await Role.findById(role);
    if (!roledata) return res.status(404).json({ error: 'Role not found' });

    const updatedUser = await User.findByIdAndUpdate(
      id,
      { name, email, role: role, isActive },
      { new: true }
    );

    if (!updatedUser) return res.status(404).json({ error: 'User not found' });

    const userWithRole = await updatedUser.populate('role');
    res.json(userWithRole);
  } catch (err) {
    res.status(500).json({ error: err.message });
  }
};


// Toggle user status
export const toggleStatus = async (req, res) => {
  const { id } = req.params;

  try {
    const user = await User.findById(id);
    if (!user) return res.status(404).json({ error: 'User not found' });

    user.isActive = !user.isActive;
    await user.save();

    res.json({ message: 'User status updated', status: user.isActive });
  } catch (err) {
    res.status(500).json({ error: err.message });
  }
};

// Delete user
export const deleteUser = async (req, res) => {
  const { id } = req.params;

  try {
    const result = await User.findByIdAndDelete(id);
    if (!result) return res.status(404).json({ error: 'User not found' });

    res.json({ message: 'User deleted' });
  } catch (err) {
    res.status(500).json({ error: err.message });
  }
};
